/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Starting up YNQ password utility
 *--------------------------------------------------------------------
 * MODULE        :
 * DEPENDENCIES  :
 ********************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

/**** for plain text to hash convertor ****/

#define NQ_MAX_BUFFER_LENGTH    255

/* open text file for update (reading and writing) */
#define NQ_FILE_READ_MODE    "r"
#define NQ_FILE_WRITE_MODE    "w"
#define NQ_FILE_UPDATE_MODE "a+"

#define NQ_FILENAME_LEN    255

typedef unsigned char UCHAR;


typedef struct{
        char   ReadLine[NQ_MAX_BUFFER_LENGTH];
        struct listelement     *link;
}listelement ;



typedef enum {
        mRV_NQ_OK,
        mRV_MISSING_CONF_FILE,
        mRV_NQ_RECORD_NOT_FOUND,
        mRV_NQ_PATH_NOT_FOUND,
        mRV_NQ_BAD_PARAMS
       }rNQRVal;

/* encryption and message digest functions */
void smbDESalgo(UCHAR *smbucResult64, UCHAR *smbucaIn64, UCHAR *smbucaKey64);
void md4(unsigned char *out, unsigned char *in, int n);

/**************************************/


/**** for password encryption tool ****/

#define END_OF_LINE_CHAR_SIZE           1
#define NQ_NULL_TERMINATOR              1
#define NQ_PWD_SALT_LEN                 5
#define NQ_PWD_USERNAMELENGTH           255 + NQ_NULL_TERMINATOR
#define NQ_PWD_MAXPASSWORDLEN           255 + NQ_NULL_TERMINATOR
#define NQ_PWD_MAX_ENCRYPTED_PASS_SIZE  (NQ_PWD_SALT_LEN * 3 + NQ_PWD_USERNAMELENGTH + NQ_PWD_MAXPASSWORDLEN)

/* size of ace-gcm key is 16 bytes */
static unsigned char key[16] = { 0x12, 0xab, 0x8b, 0xc1, 0x9d, 0x85, 0x2f, 0xf1, 0xaf, 0x5b, 0xcb, 0xc5, 0xd2, 0xae, 0xb7, 0xec};
/* size of ace-gcm nonce is 12 bytes */
static unsigned char nonce[12] = { 0x35, 0xea, 0x68, 0x51, 0x7b, 0xbe, 0xfe, 0x92, 0xaa, 0x8b, 0xcb, 0xc4};

void aesGcmEnc(const unsigned char* key, const unsigned char* nonce, const unsigned char* inText, const unsigned int textLen, unsigned char* outText);

/**************************************/

/*
 *====================================================================
 * FUNCTION  : AddItem()
 *--------------------------------------------------------------------
 * PURPOSE   :
 *--------------------------------------------------------------------
 * PARAMS    :
 * RETURNS   :
 *--------------------------------------------------------------------
 * COMMENTS  :
 *====================================================================
 */
listelement *
AddItem(
    listelement * listpointer,
    char *data
    )
{
    listelement * lp = listpointer;

    if (NULL != listpointer)
    {
        while (NULL != listpointer->link)
        {
            listpointer = (listelement *)listpointer->link;
        }

        listpointer->link = (struct listelement *)malloc(sizeof(listelement));
        listpointer = (listelement *)listpointer->link;
        listpointer->link = NULL;
        strcpy(listpointer->ReadLine, data);
        return lp;
    }
    else
    {
        listpointer = (listelement *)(struct listelement *)malloc(sizeof(listelement));
        listpointer->link = NULL;
        strcpy(listpointer->ReadLine, data);
        return listpointer;
    }
}


/*
 *====================================================================
 * FUNCTION : smbArr7toArr8()
 *--------------------------------------------------------------------
 * PURPOSE  : Converts a 56-bit DES key to 64-bit DES key
 *--------------------------------------------------------------------
 * PARAMS   : UCHAR    *ucpText        IN
 *            UCHAR     *ucpTextKey    OUT
 *
 * RETURNS  : void
 *--------------------------------------------------------------------
 * GLOBALS  : <Optional>
 *--------------------------------------------------------------------
 * COMMENTS : <Optional>
 *====================================================================
 */

void smbArr7toArr8( UCHAR *ucpText,UCHAR *ucpTextKey)
{
    int     smbwiCounter;

    ucpTextKey[0] = ucpText[0] >> 1;

    ucpTextKey[1] = ((ucpText[0] & 0x01) << 6) | (ucpText[1] >> 2);
    ucpTextKey[2] = ((ucpText[1] & 0x03) << 5) | (ucpText[2] >> 3);
    ucpTextKey[3] = ((ucpText[2] & 0x07) << 4) | (ucpText[3] >> 4);
    ucpTextKey[4] = ((ucpText[3] & 0x0F) << 3) | (ucpText[4] >> 5);
    ucpTextKey[5] = ((ucpText[4] & 0x1F) << 2) | (ucpText[5] >> 6);
    ucpTextKey[6] = ((ucpText[5] & 0x3F) << 1) | (ucpText[6] >> 7);
    ucpTextKey[7] = ucpText[6] & 0x7F;

    for (smbwiCounter = 0; smbwiCounter < 8; smbwiCounter++)
    {
        ucpTextKey[smbwiCounter] = (ucpTextKey[smbwiCounter] << 1);
    }
}

/*
 *====================================================================
 * FUNCTION : smbHashPass()
 *--------------------------------------------------------------------
 * PURPOSE  : Generates a hash password from a plain text and a key.
 *--------------------------------------------------------------------
 * PARAMS   : UCHAR    *ucpResult    OUT
 *            UCHAR     *ucpIn        IN
 *            UCHAR    *ucpKey        IN
 *
 * RETURNS  : void
 *--------------------------------------------------------------------
 * GLOBALS  : <Optional>
 *--------------------------------------------------------------------
 * COMMENTS : <Optional>
 *====================================================================
 */

 void smbHashPass(UCHAR * ucpResult ,UCHAR * ucpIn,UCHAR * ucpKey)
{
    int         smbwiCounter;
    UCHAR       smbucaIn64[64];
    UCHAR       smbucaKey64[64];
    UCHAR       smbucResult64[64];
    UCHAR       ucaOddKey[8];

    /* convert each 56-bit key to 64-bit key */
    smbArr7toArr8(ucpKey, ucaOddKey);

    /* convert the each of the 8-bytes challenge, and key arrays to */
    /* a 64-byte array                                              */
    for (smbwiCounter = 0; smbwiCounter < 64; smbwiCounter++)
    {
        smbucaIn64[smbwiCounter] = (ucpIn[smbwiCounter / 8] & (1 << (7 - (smbwiCounter % 8)))) ? 1 : 0;
        smbucaKey64[smbwiCounter] = (ucaOddKey[smbwiCounter / 8] & (1 << (7 - (smbwiCounter % 8)))) ? 1 : 0;

        /* set all the 64-byte result array to 0 */
        smbucResult64[smbwiCounter] = 0;
    }

    /* call the DES algorithm function */
    smbDESalgo(smbucResult64, smbucaIn64, smbucaKey64);

    /* fill the result array with zeros */
    for (smbwiCounter = 0; smbwiCounter < 8; smbwiCounter++)
    {
        ucpResult[smbwiCounter] = 0;
    }

    for (smbwiCounter = 0; smbwiCounter < 64; smbwiCounter++)
    {
        if (smbucResult64[smbwiCounter])
        {
            ucpResult[smbwiCounter / 8] |= (1 << (7 - (smbwiCounter % 8)));
        }
    }
}



/*
 *====================================================================
 * FUNCTION :
 *--------------------------------------------------------------------
 * PURPOSE  :
 *--------------------------------------------------------------------
 * PARAMS   :
 * RETURNS  :
 *--------------------------------------------------------------------
 * GLOBALS  : <Optional>
 *--------------------------------------------------------------------
 * COMMENTS : <Optional>
 *====================================================================
 */
int
HashPassWord(
    const unsigned char *password,
    char *hash
    )
{
    UCHAR buffer[32];                   /* 2x16 bytes password hashes */
    UCHAR ansi[15];                     /* 14 bytes null terminated uppercase ANSI password */
    short unicode[32];                  /* 31 bytes null terminated UNICODE password */
    int i, pwdlen = strlen(password);

    /* the magic key used for encryption */
    UCHAR magic[8] = {0x4b, 0x47, 0x53, 0x21, 0x40, 0x23, 0x24, 0x25};

    /* limit unicode password length by 31 character */
    if (pwdlen > sizeof(unicode) - 1)
    {
        pwdlen = sizeof(unicode) - 1;
    }

    /* clear hash and password buffers */
    memset(buffer, 0, sizeof(buffer));
    memset(ansi, 0, sizeof(ansi));
    memset(unicode, 0, sizeof(unicode));

    /* capitalize the user plain text password */
    for (i = 0; i < pwdlen; i++)
    {
        /* limit ANSI password length by 14 characters */
        if (i < sizeof(ansi) - 1)
        {
            ansi[i] = toupper(password[i]);
        }

        unicode[i] = password[i];
    }

    /* use the 14-byte password as two 56-bit keys to encrypt the magic value */
    smbHashPass(buffer, magic, ansi);
    smbHashPass(buffer + 8, magic, ansi + 7);

    /* apply MD4 to hash the unicode case sensitive password */
    md4(buffer + 16, (unsigned char *)unicode, pwdlen * 2);

    /* create output hash in a hexadecimal form */
    for (i = 0; i < 32; i++)
    {
        sprintf(&(hash[i*2]), "%02x", buffer[i]);
    }

    return mRV_NQ_OK;
}


/*
*====================================================================
* FUNCTION    : NQPlain2Hash
*--------------------------------------------------------------------
* PURPOSE    : chages all plain text passwords found in the password
*              file to hashed passwords.
*--------------------------------------------------------------------
* PARAMS    :
* RETURNS    :
*====================================================================
*/

rNQRVal
NQPlain2Hash(char * ipPasswordFile)
{
    FILE            *pFile;
    char            pctStrBuff[NQ_MAX_BUFFER_LENGTH];
    listelement     *listpointer;
    listelement     *TempPointer;
    char            *strToken;
    char            username[NQ_MAX_BUFFER_LENGTH];
    char            password[NQ_MAX_BUFFER_LENGTH];
    char            ptTempLine[NQ_MAX_BUFFER_LENGTH];
    char            actHashedPassword[NQ_MAX_BUFFER_LENGTH];
    char            UserID[32];

    if(NULL == ipPasswordFile)
    {
        return  mRV_NQ_BAD_PARAMS;
    }

    listpointer = NULL;
    /* open the file */
    pFile = fopen(ipPasswordFile, NQ_FILE_READ_MODE);

    if (NULL == pFile) /* file not found */
    {
        printf("couldn't open file for reading, the file name you specified is invalid\n");
        return mRV_MISSING_CONF_FILE; /* exit */
    }

    /*store the file in the linked list*/
    while (fgets(pctStrBuff, NQ_MAX_BUFFER_LENGTH, pFile)) /*!=NULL*/
    {
        listpointer = AddItem (listpointer, pctStrBuff);
    } /* end of while */

    TempPointer = listpointer;

    /*search the linked list for password with length < 32  */
    /*if length < 32 ==> not hashed, then hash and re-enter */
    while (NULL != TempPointer)
    {
        /*check if the entry is empty*/
        if("\n\r" == TempPointer->ReadLine)
        {
            TempPointer = (listelement *)TempPointer->link;
            continue;
        }

        strcpy(ptTempLine, TempPointer->ReadLine);
        strToken = strtok(TempPointer->ReadLine, ":\n\r");
        if(NULL == strToken)
        {
            TempPointer = (listelement *)TempPointer->link;
            continue;
        }

        strcpy (username, strToken);
        strToken = strtok(NULL, ":\n\r");
        if(NULL == strToken)
        {
            TempPointer = (listelement *)TempPointer->link;
            continue;
        }

        strcpy(password, strToken);
        strToken = strtok(NULL, ":\n\r");
        if(NULL == strToken)
        {
            TempPointer = (listelement *)TempPointer->link;
            continue;
        }

        strcpy(UserID, strToken);
        if(strlen(password) >= 32) /*already hashed*/
        {
            strcpy(TempPointer->ReadLine, ptTempLine);
            TempPointer = (listelement *)TempPointer->link;
            continue;
        }

        /*hash password*/
        HashPassWord(password, actHashedPassword);

        strcat(username, ":");
        strcat(username, actHashedPassword);
        strcat(username, ":");
        strcat(username, UserID);
        strcat(username, "\n");
        strcpy(TempPointer->ReadLine, username);
        TempPointer = (listelement *)TempPointer->link;
    }

    fclose(pFile);
    /*re-open the file to write updated data in it*/
    pFile = fopen(ipPasswordFile, NQ_FILE_WRITE_MODE);
    if (NULL == pFile)
    {
        printf("couldn't open file for writing, the file name you specified is invalid or don't have write permissions\n");
        return mRV_MISSING_CONF_FILE;
    }

    while (NULL != listpointer)
    {
           fputs(listpointer->ReadLine, pFile);
        listpointer = (listelement *)listpointer->link;
    }

    fclose(pFile);
    return mRV_NQ_OK;
}

/*
*====================================================================
* FUNCTION  : NQEncryptPassword
*--------------------------------------------------------------------
* PURPOSE   : encrypt full string of user, password and salts using
*             AES-GCM algorithm
*--------------------------------------------------------------------
* PARAMS    : IN user name
*             IN password
*             OUT full encrypted password string
*             OUT full encrypted password size (bytes)
*
* RETURNS   : None
*====================================================================
*/
static void
NQEncryptPassword(const char * userName, const char * password, unsigned char *fullEncryptedPassString, unsigned int * encryptedPassSize)
{
    int i = 0, index = 0;
    unsigned char saltBeforeUser[NQ_PWD_SALT_LEN];
    unsigned char saltafterUser[NQ_PWD_SALT_LEN];
    unsigned char saltafterPassword[NQ_PWD_SALT_LEN];
    unsigned char fullPassString[NQ_PWD_MAX_ENCRYPTED_PASS_SIZE];
    unsigned char checksum = 0;

    *encryptedPassSize = NQ_PWD_SALT_LEN + strlen(userName) + NQ_PWD_SALT_LEN + strlen(password) + NQ_PWD_SALT_LEN;

    if (*encryptedPassSize > NQ_PWD_MAX_ENCRYPTED_PASS_SIZE)
    {
        printf("encrypted Password exceeded the maximum allowed size");
        *encryptedPassSize = 0;
        return;
    }

    srand((unsigned int)time(0));
    for (i = 0; i < NQ_PWD_SALT_LEN; i++)
    {
        saltBeforeUser[i] = (unsigned char)rand();
        saltafterUser[i] = (unsigned char)rand();
        saltafterPassword[i] = (unsigned char)rand();
    }

    /* store password length in the last character of the first salt */
    saltBeforeUser[NQ_PWD_SALT_LEN - 1] = strlen(password);

    /* compose string to encrypt */
    memcpy(&fullPassString[index], saltBeforeUser, NQ_PWD_SALT_LEN);
    index += NQ_PWD_SALT_LEN;
    memcpy(&fullPassString[index], password, strlen(password));
    index += strlen(password);
    memcpy(&fullPassString[index], saltafterUser, NQ_PWD_SALT_LEN);
    index += NQ_PWD_SALT_LEN;
    memcpy(&fullPassString[index], userName, strlen(userName));
    index += strlen(userName);
    memcpy(&fullPassString[index], saltafterPassword, NQ_PWD_SALT_LEN);

    for (i = 0; i < *encryptedPassSize - 1; i++)
    {
        checksum ^= fullPassString[i];
    }

    /* store checksum in last entry */
    fullPassString[i] = checksum;

    /* performing encryption */
    aesGcmEnc(key, nonce, fullPassString, *encryptedPassSize, fullEncryptedPassString);
}


int main(int argc, char *argv[])
{
    int toolSelection;
    char dummyChar;

    printf("/*************** Visuality YNQ password utility ***************/\n");
    printf("This utility provides two services:\n\n");
    printf("1. plain text to hash convertor:\n");
    printf("   this tool converts plain text passwords to their corresponding hashes for a given\n");
    printf("   Passwords List File (called pwd.txt by default)\n\n");
    printf("2. password encryption:\n");
    printf("   this tool receives user name and password and prints to the terminal the encrypted password\n");
    printf("   the output of this tool is used by PASSWORD parameter of the Common Configuration\n");
    printf("   File (called cm_cfg.txt by default)\n\n");
    printf("Press 1 for plain text to hash convertor or 2 for password encryption tool\n");
    printf("Please type your selection:");

    if (0 >= scanf("%d", &toolSelection))
    {
        printf("wrong selection, please type 1 or 2\n");
        return 0;
    }

    /* for the enter */
    if (0 >= scanf("%c", &dummyChar))
    {
        printf("scanf was failed\n");
        return 0;
    }

    switch (toolSelection)
    {
        case 1:
        {
            char inFileName[NQ_FILENAME_LEN];
            char *p;

            printf("Please enter full file name of Password List File: ");
            fgets(inFileName, NQ_FILENAME_LEN - NQ_NULL_TERMINATOR, stdin);
            p = strrchr(inFileName, (int)'\n');
            if (NULL != p)
            {
                    p[0] = '\0';
            }

            inFileName[NQ_FILENAME_LEN - 1] = '\0';
            NQPlain2Hash(inFileName);
            break;
        };

        case 2:
        {
            char userName[NQ_PWD_USERNAMELENGTH + END_OF_LINE_CHAR_SIZE];
            char password[NQ_PWD_MAXPASSWORDLEN + END_OF_LINE_CHAR_SIZE];
            unsigned char encryptedPassword[NQ_PWD_MAX_ENCRYPTED_PASS_SIZE];
            unsigned int encryptedPasswordSize;
            char *p;
            unsigned int i;

            printf("Please enter user name (up to %d characters): ", NQ_PWD_USERNAMELENGTH - NQ_NULL_TERMINATOR);
            fgets(userName, NQ_PWD_USERNAMELENGTH + END_OF_LINE_CHAR_SIZE, stdin);
            p = strrchr(userName, (int)'\n');
            if (NULL != p)
            {
                p[0] = '\0';
            }

            userName[NQ_PWD_USERNAMELENGTH - 1] = '\0';
            printf("Please enter password to encrypt (up to %d characters): ", NQ_PWD_USERNAMELENGTH - NQ_NULL_TERMINATOR);
            fgets(password, NQ_PWD_MAXPASSWORDLEN + END_OF_LINE_CHAR_SIZE, stdin);
            p = strrchr(password, (int)'\n');
            if (NULL != p)
            {
                p[0] = '\0';
            }

            password[NQ_PWD_MAXPASSWORDLEN - 1] = '\0';
            NQEncryptPassword(userName, password, encryptedPassword, &encryptedPasswordSize);
            if (0 == encryptedPasswordSize)
            {
                puts("error while encrypting password , try again\n");
                return 0;
            }

            printf("Encrypted password: ");
            for (i = 0; i < encryptedPasswordSize; i++)
            {
                printf("%02x", encryptedPassword[i]);
            }

            printf(":E");
            printf("\nPlease copy the encrypted password above into common configuration file (called cm_cfg.txt by default)\n");
            break;
        };

        default:
        {
            printf("wrong selection, please type 1 or 2\n");
            return 0;
        };
    }

    return 1;
}
